import requests
#from datetime import datetime, UTC #python 3.11
from datetime import datetime
#from datetime import timezone  #python 3.10 and before

# Try to import UTC from datetime
try:
    from datetime import UTC
except ImportError:
    # If it fails, use timezone.utc as the fallback
    from datetime import timezone
    UTC = timezone.utc

# Created 12/02/2025, Copyright (c) 2025 by G. Stamatelatos

def fetch_weather_data(latitude, longitude, start_date, end_date, parameter, time_parameter):
    """
     Fetches weather data from the Open-Meteo API based on user inputs and stores it in structured format.

     DIRECTIONS:
     You may input the exact information of your search bellow.
     Be aware that you need to specify in the time_parameter either "daily" or "hourly".
     -------------------------------------------------------------------------------------
     For the daily parameter you may enter, depending on your situation, these strings:

     - temperature_2m_max (Celsius): Maximum daily air temperature at 2m above ground
     - temperature_2m_min (Celsius): Minimum daily air temperature at 2m above ground
     - apparent_temperature_max (Celsius): Maximum daily apparent temperature
     - apparent_temperature_min (Celsius): Minimum daily apparent temperature
     - precipitation_sum (mm): Sum of daily precipitation (including rain, showers and snowfall)
     - rain_sum (mm): Sum of daily rain
     - showers_sum (mm): Sum of daily showers
     - snowfall_sum (cm): Sum of daily snowfall
     - precipitation_hours (hours): The number of hours with rain
     - precipitation_probability_max (%): Maximum probability of precipitation
     - precipitation_probability_min (%): Minimum probability of precipitation
     - precipitation_probability_mean (%): Mean probability of precipitation
     - sunrise: Sunrise time GMT timezone
     - sunset: Sunset time GMT timezone
     - sunshine_duration (seconds): Number of seconds of sunshine per day
     - daylight_duration (seconds): Number of seconds of daylight per day
     - wind_speed_10m_max (km/h): Maximum wind speed on a day
     - wind_gusts_10m_max (km/h): Maximum wind gusts on a day
     - wind_direction_10m_dominant (Degrees): Dominant wind direction
     - shortwave_radiation_sum (MJ/m^2): The sum of solar radiation on a given day
     - et0_fao_evapotranspiration (mm): Daily sum of ET0 Reference Evapotranspiration of well watered grass field
     ---------------------------------------------------------------------------------------------------------------------
     For the hourly parameter you may enter these strings depending on your situation:

     - temperature_2m (Celsius): Air temperature at 2m above ground
     - relative_humidity_2m (%): Relative humidity at 2m above ground
     - dew_point_2m (Celsius): Dew point temperature at 2m above ground
     - apparent_temperature (Celsius): Apparent temperature is the perceived feels-like temperature
     - pressure_msl (hPa): Atmospheric air pressure reduced to mean sea level (mean sea level)
     - surface_pressure (hPa): Atmospheric air pressure at surface
     - cloud_cover (%): Total cloud cover as an area fraction
     - cloud_cover_low (%): Low level clouds and fog up to 3km altitude
     - cloud_cover_mid (%): Mid level clouds from 3 to 8 km altitude
     - cloud_cover_high (%): High level  clouds from 8 km altitude
     - wind_speed_10m, wind_speed_80m, wind_speed_120m, wind_speed_180m (km/h): Wind speed at 10, 80, 120 or 180 m above ground
     - wind_direction_10m, wind_direction_80m, wind_direction_120m, wind_direction_180m (km/h): Wind direction at 10, 80, 120 or 180 m above ground
     - wind_gusts_10m (km/h): Gusts at 10m above the ground as a maximum of the preceding hour
     - shortwave_radiation (W/m^2): Shortwaves solar radiation as average of the preceding hour
     - direct_radiation (W/m^2): Direct solar radiation as average of the preceding hour
     - direct_normal_irradiance (W/m^2): Direct solar radiation as average of the preceding hour on the normal plane (perpendicular to the sun)
     - diffuse_radiation (W/m^2): Diffuse solar radiation as average of the preceding hour
     - global_tilted_irradiance (W/m^2): Total radiation received on a tilted pane as average on the preceding hour
     - vapour_pressure_deficit (kPa): Vapour Pressure Deficit in kPa
     - cape (J/kg): Convective available potential energy
     - precipitation (mm): Total precipitation (rain, showers, snow) sum of the preceding hour
     - snowfall (cm): Snowfall amount of the preceding hour
     - precipitation_probability (%): Probability of precipitation with more than 0.1mm of the preceding hour
     - rain (mm): Rain from large scale weather systems of the preceding hour
     - showers (mm): Showers from convective precipitation in mm from the preceding hour
     - snow_depth (m): Snow depth on the ground
     - freezing_level_height (m): Altitude above sea level of the 0 deg. Celsius level
     - visibility (m): Viewing distance
     - soil_temperature_0cm, soil_temperature_6cm, soil_temperature_18cm, soil_temperature_54cm (Celsius): Temperature in the soil at 0, 6, 18 or 54 cm depths
     - soil_moisture_0_to_1cm, soil_moisture_1_to_3cm, soil_moisture_3_to_9cm, soil_moisture_9_to_27cm, soil_moisture_27_to_81cm (m^3/m^3): Average soil water content as volumetric mixing ration at 0-1, 1-3, 3-9, 9-27 or 27-81 cm depths
     - is_day: 1 if the current time step has daylight or 0 at night
     INPUT
     latitude & longitude: Coordinates for the place you are interested
     start_date: First day of interest in YYYY-MM-DD
     end_date: Last day of interest in YYYY-MM-DD
     parameter: Weather parameter to retrieve
     time_parameter: Time granularity ("hourly" or "daily")

     OUTPUT
     temperature_unit: celsius
     wind_speed_unit: km/h
     precipitation_unit: mm
     timezone: GMT

       """
    url = f"https://archive-api.open-meteo.com/v1/archive?latitude={latitude}&longitude={longitude}&start_date={start_date}&end_date={end_date}&{time_parameter}={parameter}"
    print(f"Requesting data from URL: {url}")  # Debugging: Print the URL being requested

    try:
        response = requests.get(url)
        response.raise_for_status()
        return response.json()
    except requests.exceptions.RequestException as e:
        print(f"Error fetching data: {e}")
        return None

    structured_meteo_data = {"latitude": latitude, "longitude": longitude, "time_parameter": time_parameter,
                             "data": []}

    if data and time_parameter in data and parameter in data[time_parameter]:
        structured_meteo_data["data"] = [
            {"timestamp": timestamp, "value": value}
            for timestamp, value in zip(data[time_parameter]["time"], data[time_parameter][parameter])
        ]
    else:
        print(f"No data found for parameter '{parameter}' in '{time_parameter}' data.")
        if time_parameter in data:
            print(f"Available keys in '{time_parameter}': {list(data[time_parameter].keys())}")
        else:
            print(f"'{time_parameter}' key not found in the API response.")

    return structured_meteo_data

def fetch_earthquake_data(latitude, longitude, start_date, end_date, maxradiuskm, minmagnitude, maxmagnitude):
    """
    Fetches earthquake data from the USGS API and stores it in a structured format.

     INPUT
     latitude & longitude: Coordinates for the place you are interested in
     start_date: First day of interest (YYYY-MM-DD)
     end_date: Last day of interest in (YYYY-MM-DD)
     maxradiuskm: The maximum radius in km of the search
     minmagnitude: The minimum magnitude of the earthquake(s)
     maxmagnitude: The maximum magnitude of the earthquake(s)

     OUTPUT
     coordinates
     latitude: The exact latitude of the earthquake
     longitude: The exact longitude of the earthquake
     earthquakes
     place: The region where the earthquake occurred
     magnitude: The exact magnitude of the earthquake
     time: The exact time when the earthquake occurred in the UTC timezone
    """

    url = "https://earthquake.usgs.gov/fdsnws/event/1/query"
    print(f"Requesting data from URL: {url}")  # Debugging: Print the URL being requested
    params = {
        "format": "geojson",
        "latitude": latitude,
        "longitude": longitude,
        "maxradiuskm": maxradiuskm,
        "starttime": start_date,
        "endtime": end_date,
        "minmagnitude": minmagnitude,
        "maxmagnitude": maxmagnitude
    }

    try:
        response = requests.get(url, params=params)
        response.raise_for_status()
        data = response.json()

        structured_earthquake_data = {"latitude": latitude, "longitude": longitude, "earthquakes": []}

        if data.get("features"):
            for event in data["features"]:
                properties = event["properties"]
                geometry = event["geometry"]
                structured_earthquake_data["earthquakes"].append({
                    "place": properties["place"],
                    "magnitude": properties["mag"],
                    "time": datetime.fromtimestamp(properties["time"] / 1000, tz=UTC).strftime('%Y-%m-%d %H:%M:%S'),
                    "coordinates": {
                        "latitude": geometry["coordinates"][1],
                        "longitude": geometry["coordinates"][0]
                    }
                })
        else:
            print("No earthquakes found in the specified area and time range.")

        return structured_earthquake_data
    except requests.exceptions.RequestException as e:
        print(f"Failed to fetch earthquake data: {e}")
        return None

"""
Example:

Lets say we want to calculate the hourly wind gusts and
earthquakes within 50 km radius with minimum magnitude 3 and maximum magnitude 6
in Santorini from 05-02-2025 to 07-02-2025.
Santorini coordinates:
latitude: 36.43389
longitude: 25.33778

"""


# For the example
def main():

    # Fetch weather data
    print("Fetching weather data for Santorini...")
    structured_meteo_data = fetch_weather_data("36.43389", "25.33778", "2025-02-05", "2025-02-07", "wind_gusts_10m","hourly")
    if structured_meteo_data:
        print("Santorini's Weather Data:")
        #print(structured_meteo_data)
    else:
        print("Failed to fetch weather data.")


    # Fetch earthquake data
    print("Fetching earthquake data for Santorini...")
    #structured_earthquake_data = fetch_earthquake_data("36.43389", "25.33778", "2025-02-05", "2025-02-27", "50", "3", "6")
    structured_earthquake_data = fetch_earthquake_data("36.435373", "28.217828", "2024-01-01", "2024-10-04", "50", "3", "6")

    if structured_earthquake_data:
        print("Santorini's Earthquake Data:")
        print(structured_earthquake_data)
    else:
        print("Failed to fetch earthquake data.")


    return structured_meteo_data, structured_earthquake_data


if __name__ == "__main__":
    structured_meteo_data, structured_earthquake_data = main()


